document.addEventListener('DOMContentLoaded', function () {
    window.editor = ace.edit("editor");
    editor.setTheme("ace/theme/tomorrow_night_eighties");
    editor.session.setMode("ace/mode/lua");
    editor.setShowPrintMargin(false);
    editor.setOptions({
        fontSize: "12px",
        fontFamily: "Monaco, Menlo, 'Ubuntu Mono', Consolas, source-code-pro, monospace",
        enableBasicAutocompletion: true,
        enableSnippets: true,
        enableLiveAutocompletion: true,
    });

    let tabCount = 1;
    let activeTabId = 1;
    const tabs = document.getElementById('tabs');
    const addTabButton = document.getElementById('add-tab');

    let tabContents = {};

    function adjustTabWidths() {
        const allTabs = document.querySelectorAll('.tab');
        if (allTabs.length >= 10) {
            const calculatedWidth = Math.max(70, Math.min(80, 70 + (allTabs.length >= 10 ? 10 : 0)));
            allTabs.forEach(tab => {
                const nameEl = tab.querySelector('.tab-name');
                if (nameEl) {
                    nameEl.style.fontSize = '12px';
                }
                tab.classList.add('narrow');
            });
        } else {
            allTabs.forEach(tab => {
                const nameEl = tab.querySelector('.tab-name');
                if (nameEl) {
                    nameEl.style.fontSize = '14px';
                }
                tab.classList.remove('narrow');
            });
        }
    }


    addTabButton.addEventListener('click', function () {
        addNewTab();
    });

    function addNewTab() {
        tabCount++;
        const newTabId = tabCount;
        const newTab = document.createElement('div');
        newTab.className = 'tab';
        newTab.dataset.tabId = newTabId;
        newTab.innerHTML = `
            <span class="tab-name">Script ${newTabId}</span>
            <button class="remove-tab">×</button>
        `;
        tabs.appendChild(newTab);
        tabContents[newTabId] = '';
        adjustTabWidths();
        switchTab(newTabId);
        saveTabs();
    }

    const luauFunctions = [
        { name: "print", value: "print", score: 1000, meta: "function" },
        { name: "pairs", value: "pairs", score: 990, meta: "function" },
        { name: "ipairs", value: "ipairs", score: 985, meta: "function" },
        { name: "next", value: "next", score: 980, meta: "function" },
        { name: "type", value: "type", score: 975, meta: "function" },
        { name: "tostring", value: "tostring", score: 970, meta: "function" },
        { name: "tonumber", value: "tonumber", score: 965, meta: "function" },
        { name: "table.insert", value: "table.insert", score: 960, meta: "function" },
        { name: "table.remove", value: "table.remove", score: 955, meta: "function" },
        { name: "table.sort", value: "table.sort", score: 950, meta: "function" },
        { name: "math.abs", value: "math.abs", score: 940, meta: "function" },
        { name: "math.floor", value: "math.floor", score: 935, meta: "function" },
        { name: "math.ceil", value: "math.ceil", score: 930, meta: "function" },
        { name: "math.random", value: "math.random", score: 925, meta: "function" },
        { name: "string.format", value: "string.format", score: 920, meta: "function" },
        { name: "string.sub", value: "string.sub", score: 915, meta: "function" },
        { name: "string.find", value: "string.find", score: 910, meta: "function" },
        { name: "workspace", value: "workspace", score: 900, meta: "service" },
        { name: "game", value: "game", score: 900, meta: "service" },
        { name: "script", value: "script", score: 900, meta: "variable" },
        { name: "Instance.new", value: "Instance.new", score: 895, meta: "function" },
        { name: "Part", value: "Part", score: 890, meta: "class" },
        { name: "Model", value: "Model", score: 885, meta: "class" },
        { name: "Humanoid", value: "Humanoid", score: 880, meta: "class" },
        { name: "CFrame", value: "CFrame", score: 875, meta: "class" },
        { name: "FindFirstChild", value: "FindFirstChild", score: 870, meta: "method" },
        { name: "WaitForChild", value: "WaitForChild", score: 865, meta: "method" },
        { name: "Destroy", value: "Destroy", score: 860, meta: "method" },
        { name: "Clone", value: "Clone", score: 855, meta: "method" },
        { name: "Touched", value: "Touched", score: 850, meta: "event" },
        { name: "Changed", value: "Changed", score: 845, meta: "event" },
        { name: "AncestryChanged", value: "AncestryChanged", score: 840, meta: "event" },
        { name: "Enum", value: "Enum", score: 835, meta: "enum" },
        { name: "Enum.Material", value: "Enum.Material", score: 830, meta: "enum" },
        { name: "Enum.Font", value: "Enum.Font", score: 825, meta: "enum" },
        { name: "assert", value: "assert", score: 900, meta: "function" },
        { name: "collectgarbage", value: "collectgarbage", score: 890, meta: "function" },
        { name: "coroutine.create", value: "coroutine.create", score: 880, meta: "function" },
        { name: "coroutine.resume", value: "coroutine.resume", score: 875, meta: "function" },
        { name: "coroutine.running", value: "coroutine.running", score: 870, meta: "function" },
        { name: "coroutine.status", value: "coroutine.status", score: 865, meta: "function" },
        { name: "coroutine.wrap", value: "coroutine.wrap", score: 860, meta: "function" },
        { name: "dofile", value: "dofile", score: 855, meta: "function" },
        { name: "error", value: "error", score: 850, meta: "function" },
        { name: "getmetatable", value: "getmetatable", score: 845, meta: "function" },
        { name: "load", value: "load", score: 840, meta: "function" },
        { name: "loadfile", value: "loadfile", score: 835, meta: "function" },
        { name: "loadstring", value: "loadstring", score: 830, meta: "function" },
        { name: "next", value: "next", score: 825, meta: "function" },
        { name: "pcall", value: "pcall", score: 820, meta: "function" },
        { name: "rawequal", value: "rawequal", score: 815, meta: "function" },
        { name: "rawget", value: "rawget", score: 810, meta: "function" },
        { name: "rawlen", value: "rawlen", score: 805, meta: "function" },
        { name: "rawset", value: "rawset", score: 800, meta: "function" },
        { name: "require", value: "require", score: 795, meta: "function" },
        { name: "select", value: "select", score: 790, meta: "function" },
        { name: "setmetatable", value: "setmetatable", score: 785, meta: "function" },
        { name: "tonumber", value: "tonumber", score: 780, meta: "function" },
        { name: "tostring", value: "tostring", score: 775, meta: "function" },
        { name: "type", value: "type", score: 770, meta: "function" },
        { name: "xpcall", value: "xpcall", score: 765, meta: "function" },
        { name: "tick", value: "tick", score: 760, meta: "function" },
        { name: "time", value: "time", score: 755, meta: "function" },
        { name: "wait", value: "wait", score: 750, meta: "function" },
        { name: "spawn", value: "spawn", score: 745, meta: "function" },
        { name: "delay", value: "delay", score: 740, meta: "function" },
        { name: "typeof", value: "typeof", score: 735, meta: "function" },
        { name: "bit32.band", value: "bit32.band", score: 730, meta: "function" },
        { name: "bit32.bnot", value: "bit32.bnot", score: 725, meta: "function" },
        { name: "bit32.bor", value: "bit32.bor", score: 720, meta: "function" },
        { name: "bit32.btest", value: "bit32.btest", score: 715, meta: "function" },
        { name: "bit32.bxor", value: "bit32.bxor", score: 710, meta: "function" },
        { name: "bit32.lrotate", value: "bit32.lrotate", score: 705, meta: "function" },
        { name: "bit32.lshift", value: "bit32.lshift", score: 700, meta: "function" },
        { name: "bit32.rrotate", value: "bit32.rrotate", score: 695, meta: "function" },
        { name: "bit32.rshift", value: "bit32.rshift", score: 690, meta: "function" },
        { name: "utf8.char", value: "utf8.char", score: 685, meta: "function" },
        { name: "utf8.codes", value: "utf8.codes", score: 680, meta: "function" },
        { name: "utf8.codepoint", value: "utf8.codepoint", score: 675, meta: "function" },
        { name: "utf8.len", value: "utf8.len", score: 670, meta: "function" },
        { name: "utf8.offset", value: "utf8.offset", score: 665, meta: "function" },
        { name: "coroutine.yield", value: "coroutine.yield", score: 660, meta: "function" },
        { name: "debug.debug", value: "debug.debug", score: 655, meta: "function" },
        { name: "debug.gethook", value: "debug.gethook", score: 650, meta: "function" },
        { name: "debug.getinfo", value: "debug.getinfo", score: 645, meta: "function" },
        { name: "debug.getlocal", value: "debug.getlocal", score: 640, meta: "function" },
        { name: "debug.getmetatable", value: "debug.getmetatable", score: 635, meta: "function" },
        { name: "debug.getregistry", value: "debug.getregistry", score: 630, meta: "function" },
        { name: "debug.getupvalue", value: "debug.getupvalue", score: 625, meta: "function" },
        { name: "debug.sethook", value: "debug.sethook", score: 620, meta: "function" },
        { name: "debug.setlocal", value: "debug.setlocal", score: 615, meta: "function" },
        { name: "debug.setupvalue", value: "debug.setupvalue", score: 610, meta: "function" },
        { name: "debug.traceback", value: "debug.traceback", score: 605, meta: "function" },
        { name: "os.clock", value: "os.clock", score: 600, meta: "function" },
        { name: "os.date", value: "os.date", score: 595, meta: "function" },
        { name: "os.difftime", value: "os.difftime", score: 590, meta: "function" },
        { name: "os.execute", value: "os.execute", score: 585, meta: "function" },
        { name: "os.exit", value: "os.exit", score: 580, meta: "function" },
        { name: "os.getenv", value: "os.getenv", score: 575, meta: "function" },
        { name: "os.remove", value: "os.remove", score: 570, meta: "function" },
        { name: "os.rename", value: "os.rename", score: 565, meta: "function" },
        { name: "os.setlocale", value: "os.setlocale", score: 560, meta: "function" },
        { name: "os.time", value: "os.time", score: 555, meta: "function" },
        { name: "os.tmpname", value: "os.tmpname", score: 550, meta: "function" },
        { name: "math.acos", value: "math.acos", score: 545, meta: "function" },
        { name: "math.asin", value: "math.asin", score: 540, meta: "function" },
        { name: "math.atan", value: "math.atan", score: 535, meta: "function" },
        { name: "math.atan2", value: "math.atan2", score: 530, meta: "function" },
        { name: "math.cos", value: "math.cos", score: 525, meta: "function" },
        { name: "math.cosh", value: "math.cosh", score: 520, meta: "function" },
        { name: "math.deg", value: "math.deg", score: 515, meta: "function" },
        { name: "math.exp", value: "math.exp", score: 510, meta: "function" },
        { name: "math.frexp", value: "math.frexp", score: 505, meta: "function" },
        { name: "math.ldexp", value: "math.ldexp", score: 500, meta: "function" },
        { name: "math.log", value: "math.log", score: 495, meta: "function" },
        { name: "math.log10", value: "math.log10", score: 490, meta: "function" },
        { name: "math.max", value: "math.max", score: 485, meta: "function" },
        { name: "math.min", value: "math.min", score: 480, meta: "function" },
        { name: "math.modf", value: "math.modf", score: 475, meta: "function" },
        { name: "math.pow", value: "math.pow", score: 470, meta: "function" },
        { name: "math.rad", value: "math.rad", score: 465, meta: "function" },
        { name: "math.sin", value: "math.sin", score: 460, meta: "function" },
        { name: "math.sinh", value: "math.sinh", score: 455, meta: "function" },
        { name: "math.tan", value: "math.tan", score: 450, meta: "function" },
        { name: "math.tanh", value: "math.tanh", score: 445, meta: "function" },
        { name: "string.byte", value: "string.byte", score: 440, meta: "function" },
        { name: "string.charpattern", value: "string.charpattern", score: 435, meta: "function" },
        { name: "string.gmatch", value: "string.gmatch", score: 430, meta: "function" },
        { name: "string.gsub", value: "string.gsub", score: 425, meta: "function" },
        { name: "string.match", value: "string.match", score: 420, meta: "function" },
        { name: "string.pack", value: "string.pack", score: 415, meta: "function" },
        { name: "string.packsize", value: "string.packsize", score: 410, meta: "function" },
        { name: "string.rep", value: "string.rep", score: 405, meta: "function" },
        { name: "string.reverse", value: "string.reverse", score: 400, meta: "function" },
        { name: "string.sub", value: "string.sub", score: 395, meta: "function" },
        { name: "string.unpack", value: "string.unpack", score: 390, meta: "function" },
        { name: "table.concat", value: "table.concat", score: 385, meta: "function" },
        { name: "table.maxn", value: "table.maxn", score: 380, meta: "function" },
        { name: "table.move", value: "table.move", score: 375, meta: "function" },
        { name: "table.pack", value: "table.pack", score: 370, meta: "function" },
        { name: "table.unpack", value: "table.unpack", score: 365, meta: "function" },
        { name: "os.clock", value: "os.clock", score: 360, meta: "function" },
        { name: "os.date", value: "os.date", score: 355, meta: "function" },
        { name: "os.time", value: "os.time", score: 350, meta: "function" },
    ];

    const luauCompleter = {
        getCompletions: function (editor, session, pos, prefix, callback) {
            if (prefix.length === 0) {
                callback(null, []);
                return;
            }
            callback(null, luauFunctions.map(word => ({
                caption: word.name,
                value: word.value,
                score: word.score,
                meta: word.meta
            })));
        }
    };

    ace.require("ace/ext/language_tools").addCompleter(luauCompleter);


    function saveTabs() {
        localStorage.setItem('tabsData', JSON.stringify(tabContents));
    }

    function loadTabs() {
        const saved = localStorage.getItem('tabsData');
        if (saved) {
            tabContents = JSON.parse(saved);
            while (tabs.firstChild) {
                tabs.removeChild(tabs.firstChild);
            }
            for (const tabId in tabContents) {
                createTabElement(tabId, tabContents[tabId]);
            }
            if (Object.keys(tabContents).length > 0) {
                activeTabId = parseInt(Object.keys(tabContents)[0]);
                switchTab(activeTabId);
                tabCount = Math.max(...Object.keys(tabContents).map(x => parseInt(x)));
            }
        }
    }

    function createTabElement(tabId, content) {
        const newTab = document.createElement('div');
        newTab.className = 'tab';
        newTab.dataset.tabId = tabId;
        newTab.innerHTML = `
            <span class="tab-name">Script ${tabId}</span>
            <button class="remove-tab">×</button>
        `;
        tabs.appendChild(newTab);
        newTab.querySelector('.remove-tab').addEventListener('click', e => {
            e.stopPropagation();
            removeTab(parseInt(tabId));
        });
        newTab.addEventListener('click', () => {
            switchTab(parseInt(tabId));
        });
        adjustTabWidths();
    }

    window.onload = () => {
        loadTabs();
        if (Object.keys(tabContents).length === 0) {
            tabContents[1] = editor.getValue();
            createTabElement(1, tabContents[1]);
            switchTab(1);
        }
    };

    tabs.addEventListener('click', function(e) {
        const tab = e.target.closest('.tab');
        if (!tab) return;
        if (e.target.classList.contains('remove-tab')) {
            const tabId = parseInt(tab.dataset.tabId);
            removeTab(tabId);
            return;
        }
        const tabId = parseInt(tab.dataset.tabId);
        switchTab(tabId);
    });

    function switchTab(tabId) {
        if (tabContents.hasOwnProperty(activeTabId) && activeTabId !== tabId) {
            tabContents[activeTabId] = editor.getValue();
        }
        activeTabId = tabId;
        document.querySelectorAll('.tab').forEach(tab => {
            tab.classList.remove('active');
        });
        const activeTab = document.querySelector(`.tab[data-tab-id="${tabId}"]`);
        if (activeTab) activeTab.classList.add('active');
        editor.setValue(tabContents[tabId] || '', -1);
        saveTabs();
    }

    function removeTab(tabId) {
        tabCount--;
        if (document.querySelectorAll('.tab').length <= 1) return;
        const tabElement = document.querySelector(`.tab[data-tab-id="${tabId}"]`);
        if (tabElement) tabElement.remove();
        delete tabContents[tabId];
        if (activeTabId === tabId) {
            const firstTab = document.querySelector('.tab');
            if (firstTab) switchTab(parseInt(firstTab.dataset.tabId));
        }
        adjustTabWidths();
        saveTabs();
    }

    editor.on('change', function() {
        tabContents[activeTabId] = editor.getValue();
        saveTabs();
    });

    window.TabEditor = {
        SetTheme: function(theme) {
            editor.setTheme(`ace/theme/${theme}`);
        },
        GetText: function() {
            return editor.getValue();
        },
        SetText: function(text) {
            editor.setValue(text, -1);
            tabContents[activeTabId] = text;
        }
    };
});
